<?php

namespace AppBundle\Service;

use AppBundle\Entity\Application;
use AppBundle\Entity\User;
use Doctrine\ORM\EntityManager;
use Swift_Mailer;
use Symfony\Bundle\TwigBundle\TwigEngine;

class MailerService
{
    /**
     * @var Swift_Mailer
     */
    private $mailer;

    /**
     * @var TwigEngine
     */
    private $templating;

    /**
     * @var EntityManager
     */
    private $entityManager;

    /**
     * @var string
     */
    private $mailerUser;

    /**
     * @param Swift_Mailer $mailer
     * @param TwigEngine $templating
     * @param EntityManager $entityManager
     * @internal param $mailerUser
     */
    public function __construct(
        Swift_Mailer $mailer,
        TwigEngine $templating,
        EntityManager $entityManager,
        $mailerUser
    ) {
        $this->mailer = $mailer;
        $this->templating = $templating;
        $this->entityManager = $entityManager;
        $this->mailerUser = $mailerUser;
    }

    /**
     * @param User $user
     * @param $password
     */
    public function sendNewUserEmail(User $user, $password)
    {
        $this->sendEmail(
            'Dane dostępu do twojego konta',
            $user->getEmail(),
            $this->templating->render(
                'AppBundle:Mailer:new_user.html.twig',
                array(
                    'user' => $user,
                    'password' => $password
                )
            )
        );
    }

    /**
     * @param User $user
     */
    public function sendRecoveryLinkEmail(User $user)
    {
        $this->sendEmail(
            'Link resetujący hasło',
            $user->getEmail(),
            $this->templating->render(
                'AppBundle:Mailer:recovery_link.html.twig',
                array(
                    'user' => $user
                )
            )
        );
    }

    /**
     * @param User $user
     * @param $password
     */
    public function sendRecoveryPasswordEmail(User $user, $password)
    {
        $this->sendEmail(
            'Nowe hasło',
            $user->getEmail(),
            $this->templating->render(
                'AppBundle:Mailer:recovery_password.html.twig',
                array(
                    'user' => $user,
                    'password' => $password
                )
            )
        );
    }

    /**
     * @param $result
     */
    public function sendXMLRatesFileDownloadError($result)
    {
        $this->sendEmail(
            'Błąd podczas pobierania pliku XML',
            $this->getAdminsEmails(),
            $this->templating->render(
                'AppBundle:Mailer:xml_rates_file_download_error.html.twig',
                array(
                    'result' => $result
                )
            )
        );
    }

    /**
     * @param Application $application
     * @throws \Exception
     * @throws \Twig_Error
     */
    public function sendNewOrderAdmin(Application $application)
    {
        $this->sendEmail(
            'Wiadomość z DrobnicaMorska.pl',
            $this->getAdminsEmails(),
            $this->templating->render(
                'AppBundle:Mailer:new_order.html.twig',
                array(
                    'application' => $application
                )
            )
        );
    }

    /**
     * @param Application $application
     * @throws \Exception
     * @throws \Twig_Error
     */
    public function sendNewOrderConsultant(Application $application)
    {
        $this->sendEmail(
            'Wiadomość z DrobnicaMorska.pl',
            $this->getConsultantsEmails(),
            $this->templating->render(
                'AppBundle:Mailer:new_order.html.twig',
                array(
                    'application' => $application
                )
            )
        );
    }

    /**
     * @param Application $application
     * @throws \Exception
     * @throws \Twig_Error
     */
    public function sendNewOrderUser(Application $application)
    {
        $this->sendEmail(
            'Wiadomość z DrobnicaMorska.pl',
            $application->getEmail(),
            $this->templating->render(
                'AppBundle:Mailer:new_order.html.twig',
                array(
                    'application' => $application
                )
            )
        );
    }

    /**
     * @param $subject
     * @param $to
     * @param $body
     * @param null $from
     */
    private function sendEmail($subject, $to, $body, $from = null)
    {
        $from = $from ?: $this->mailerUser;

        $message = \Swift_Message::newInstance()
            ->setSubject($subject)
            ->setFrom($from)
            ->setBcc($to)
            ->setBody($body, 'text/html');

        $this->mailer->send($message);
    }

    /**
     * @return \AppBundle\Entity\User[]
     */
    private function getAdminsEmails()
    {
        $admins = $this->entityManager->getRepository('AppBundle:User')->findNotRemoved();
        $adminsEmails = array();

        foreach ($admins as $admin) {
            $adminsEmails[] = $admin->getEmail();
        }

        return $adminsEmails;
    }

    /**
     * @return \AppBundle\Entity\Consultant[]
     */
    private function getConsultantsEmails()
    {
        $consultants = $this->entityManager->getRepository('AppBundle:Consultant')->findNotRemoved();
        $consultantsEmails = array();

        foreach ($consultants as $consultant) {
            $consultantsEmails[] = $consultant->getEmail();
        }

        return $consultantsEmails;
    }
}