<?php

namespace AppBundle\Controller\Admin;

use AppBundle\Entity\User;
use AppBundle\Entity\UserHistory;
use AppBundle\Form\Type\UserType;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;

class UserController extends Controller
{
    /**
     * @Template()
     *
     * @param Request $request
     * @return array
     */
    public function listAction(Request $request)
    {
        $em = $this->getDoctrine();
        $query = $em->getRepository('AppBundle:User')->getListQuery();

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $query,
            $request->query->get('page', 1),
            20
        );

        return array(
            'pagination' => $pagination
        );
    }

    /**
     * @param Request $request
     * @return array
     */
    public function createAction(Request $request)
    {
        $user = new User();
        $form = $this->createForm(new UserType(), $user);
        $form->handleRequest($request);

        if ($form->isValid()) {
            $password = $this->getPasswordService()->setRandomPassword($user);
            $user->setRoles(array('ROLE_ADMIN'));

            $em = $this->getDoctrine()->getManager();
            $em->persist($user);
            $em->flush();

            $userHistory = new UserHistory();
            $userHistory
                ->setEmail($user->getEmail())
                ->setFullName($user->getFullName())
                ->setEditor($this->getUser())
                ->setUser($user);

            $em->persist($userHistory);
            $em->flush();

            $mailerService = $this->getMailerService();
            $mailerService->sendNewUserEmail($user, $password);

            return $this->redirect($this->generateUrl('app_admin.user.list'));
        }

        return $this->render(
            'AppBundle:Admin/User:form.html.twig',
            array(
                'form' => $form->createView()
            )
        );
    }

    /**
     * @param Request $request
     * @param User $user
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, User $user)
    {
        if($user->getIsRemoved()) {
            throw new AccessDeniedException();
        }

        $userClone = clone($user);
        $isUserAccount = $user == $this->getUser();
        $form = $this->createForm(new UserType($isUserAccount), $user);
        $form->handleRequest($request);

        if ($form->isValid()) {
            if(isset($form['password'])) {
                $actualPassword = $form['password']->getData();
                $newPassword = $form['newPassword']->getData();

                if($isUserAccount && $actualPassword && $newPassword) {
                    $passwordService = $this->getPasswordService();
                    $passwordService->setEncodedPassword($user, $newPassword);
                } else {
                    $user->setPassword($userClone->getPassword());
                }
            }

            $this->saveUserEditHistory($userClone, $user);

            $em = $this->getDoctrine()->getManager();
            $em->flush();

            return $this->redirect($this->generateUrl('app_admin.user.list'));
        }

        return $this->render(
            'AppBundle:Admin/User:form.html.twig',
            array(
                'user' => $user,
                'form' => $form->createView()
            )
        );
    }

    /**
     * @param User $user
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function removeAction(User $user)
    {
        if($user->getIsRemoved()) {
            throw new AccessDeniedException();
        }

        $em = $this->getDoctrine()->getManager();
        $user->setIsRemoved(true);
        $user->setEmail(null);

        $userHistory = new UserHistory();
        $userHistory
            ->setEditor($this->getUser())
            ->setUser($user)
            ->setIsRemoved(true);

        $em->persist($userHistory);
        $em->flush();

        return $this->redirect($this->generateUrl('app_admin.user.list'));
    }

    /**
     * @Template()
     *
     * @param Request $request
     * @param User $user
     * @return array
     */
    public function historyAction(Request $request, User $user)
    {
        $em = $this->getDoctrine();
        $query = $em->getRepository('AppBundle:UserHistory')->getHistoryListQuery($user);

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $query,
            $request->query->get('page', 1),
            20,
            array('defaultSortFieldName' => 'uh.createdAt', 'defaultSortDirection' => 'desc')
        );

        return array(
            'pagination' => $pagination
        );
    }

    /**
     * @Template()
     *
     * @param Request $request
     * @return array
     */
    public function removedListAction(Request $request)
    {
        $em = $this->getDoctrine();
        $query = $em->getRepository('AppBundle:User')->getRemovedListQuery();

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $query,
            $request->query->get('page', 1),
            20,
            array('defaultSortFieldName' => 'u.updatedAt', 'defaultSortDirection' => 'desc')
        );

        return array(
            'pagination' => $pagination
        );
    }

    /**
     * @param User $oldUser
     * @param User $user
     */
    private function saveUserEditHistory(User $oldUser, User $user)
    {
        $isEdited = false;
        $userHistory = new UserHistory();

        if($oldUser->getFullName() != $user->getFullName()) {
            $userHistory->setFullName($user->getFullName());
            $isEdited = true;
        }

        if($oldUser->getEmail() != $user->getEmail()) {
            $userHistory->setEmail($user->getEmail());
            $isEdited = true;
        }

        if($isEdited) {
            $em = $this->getDoctrine()->getManager();
            $userHistory
                ->setEditor($this->getUser())
                ->setUser($user);

            $em->persist($userHistory);
            $em->flush();
        }
    }

    /**
     * @return \AppBundle\Service\PasswordService
     */
    private function getPasswordService()
    {
        return $this->get('app.service.password');
    }

    /**
     * @return \AppBundle\Service\MailerService
     */
    private function getMailerService()
    {
        return $this->get('app.service.mailer');
    }
}
