<?php

namespace AppBundle\Controller\Admin;

use AppBundle\Entity\Consultant;
use AppBundle\Entity\ConsultantHistory;
use AppBundle\Form\Type\ConsultantType;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;

class ConsultantController extends Controller
{
    /**
     * @Template()
     *
     * @param Request $request
     * @return array
     */
    public function listAction(Request $request)
    {
        $em = $this->getDoctrine();
        $query = $em->getRepository('AppBundle:Consultant')->getListQuery();

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $query,
            $request->query->get('page', 1),
            20,
            array('defaultSortFieldName' => 'c.email', 'defaultSortDirection' => 'desc')
        );

        return array(
            'pagination' => $pagination
        );
    }

    /**
     * @param Request $request
     * @return array
     */
    public function createAction(Request $request)
    {
        $consultant = new Consultant();
        $form = $this->createForm(new ConsultantType(), $consultant);
        $form->handleRequest($request);

        if ($form->isValid()) {
            $em = $this->getDoctrine()->getManager();
            $em->persist($consultant);

            $consultantHistory = new ConsultantHistory();
            $consultantHistory
                ->setEmail($consultant->getEmail())
                ->setUser($this->getUser())
                ->setConsultant($consultant);

            $em->persist($consultantHistory);
            $em->flush();

            return $this->redirect($this->generateUrl('app_admin.consultant.list'));
        }

        return $this->render(
            'AppBundle:Admin/Consultant:form.html.twig',
            array(
                'form' => $form->createView()
            )
        );
    }

    /**
     * @param Request $request
     * @param Consultant $consultant
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, Consultant $consultant)
    {
        if($consultant->getIsRemoved()) {
            throw new AccessDeniedException();
        }

        $consultantClone = clone($consultant);
        $form = $this->createForm(new ConsultantType(), $consultant);
        $form->handleRequest($request);

        if ($form->isValid()) {
            $em = $this->getDoctrine()->getManager();
            $em->flush();

            $this->saveConsultantEditHistory($consultantClone, $consultant);

            return $this->redirect($this->generateUrl('app_admin.consultant.list'));
        }

        return $this->render(
            'AppBundle:Admin/Consultant:form.html.twig',
            array(
                'consultant' => $consultant,
                'form' => $form->createView()
            )
        );
    }

    /**
     * @param Consultant $consultant
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function removeAction(Consultant $consultant)
    {
        if($consultant->getIsRemoved()) {
            throw new AccessDeniedException();
        }

        $em = $this->getDoctrine()->getManager();
        $consultant->setIsRemoved(true);

        $consultantHistory = new ConsultantHistory();
        $consultantHistory
            ->setUser($this->getUser())
            ->setConsultant($consultant)
            ->setIsRemoved(true);

        $em->persist($consultantHistory);
        $em->flush();

        return $this->redirect($this->generateUrl('app_admin.consultant.list'));
    }

    /**
     * @Template()
     *
     * @param Request $request
     * @param Consultant $consultant
     * @return array
     */
    public function historyAction(Request $request, Consultant $consultant)
    {
        $em = $this->getDoctrine();
        $query = $em->getRepository('AppBundle:ConsultantHistory')->getHistoryListQuery($consultant);

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $query,
            $request->query->get('page', 1),
            20,
            array('defaultSortFieldName' => 'ch.createdAt', 'defaultSortDirection' => 'desc')
        );

        return array(
            'pagination' => $pagination
        );
    }

    /**
     * @Template()
     *
     * @param Request $request
     * @return array
     */
    public function removedListAction(Request $request)
    {
        $em = $this->getDoctrine();
        $query = $em->getRepository('AppBundle:Consultant')->getRemovedListQuery();

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $query,
            $request->query->get('page', 1),
            20,
            array('defaultSortFieldName' => 'c.updatedAt', 'defaultSortDirection' => 'desc')
        );

        return array(
            'pagination' => $pagination
        );
    }

    /**
     * @param Consultant $oldConsultant
     * @param Consultant $consultant
     */
    private function saveConsultantEditHistory(Consultant $oldConsultant, Consultant $consultant)
    {
        $isEdited = false;
        $consultantHistory = new ConsultantHistory();

        if($oldConsultant->getEmail() != $consultant->getEmail()) {
            $consultantHistory->setEmail($consultant->getEmail());
            $isEdited = true;
        }

        if($isEdited) {
            $em = $this->getDoctrine()->getManager();
            $consultantHistory
                ->setUser($this->getUser())
                ->setConsultant($consultant);

            $em->persist($consultantHistory);
            $em->flush();
        }
    }
}
