<?php

namespace AppBundle\Command;

use AppBundle\Controller\Admin\ConnectionController;
use AppBundle\Entity\Connection;
use AppBundle\Entity\ConnectionHistory;
use AppBundle\Entity\Port;
use AppBundle\Entity\PortHistory;
use DateTime;
use Exception;
use SimpleXMLElement;
use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

class XMLRatesFileDownloadCommand extends ContainerAwareCommand
{
    /**
     * @var Port[]
     */
    private $ports;

    protected function configure()
    {
        $this
            ->setName('rates:download')
            ->setDescription('Downloads rates from XML file');
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $xmlFile = $this->downloadXMLFile();
        if ($xmlFile instanceof Exception) {
            $this->sendErrorMail('Błąd przy pobieraniu pliku XML. <br>' . $xmlFile->getMessage());

            return;
        }

        $xmlElement = $this->loadXMLFromFile($xmlFile);
        if ($xmlElement instanceof Exception) {
            $this->sendErrorMail('Błędny format pliku XML. <br>' . $xmlElement->getMessage());

            return;
        }

        $rates = $this->loadRatesFromXMLElement($xmlElement);
        if ($xmlElement instanceof Exception) {
            $this->sendErrorMail('Błędny format pliku XML. Brak klucza "rate".');

            return;
        }

        $result = $this->validateRates($rates);
        if (count($result['valid']) > 0) {
            $result['edited'] = $this->saveRatesToDatabase($result['valid']);
        }

        if (//count($result['valid']) == 0 ||
            //$result['edited'] < count($result['valid']) ||
            count($result['fieldsNumberErrorIds']) ||
            count($result['fieldsKeysErrorIds']) ||
            count($result['dateErrorIds'])
        ) {
            $this->sendErrorMail($result);
        }
    }

    /**
     * @return Exception|string
     */
    private function downloadXMLFile()
    {
        $server = $this->getContainer()->getParameter('xml_rates_server');
        $login = $this->getContainer()->getParameter('xml_rates_login');
        $password = $this->getContainer()->getParameter('xml_rates_password');
        $directoryName = $this->getContainer()->getParameter('xml_rates_directory_name');
        $fileName = $this->getContainer()->getParameter('xml_rates_file_name');
        $filePath = $directoryName . DIRECTORY_SEPARATOR . $fileName;

        try {
            $connection = ftp_connect($server);
            ftp_login($connection, $login, $password);

            // download
            ob_start();
            ftp_get($connection, "php://output", $filePath, FTP_BINARY);
            $file = ob_get_contents();
            ob_end_clean();

            // rename
            $now = new \DateTime();
            $newFileName = str_replace('.xml', $now->format('Ymd') . '.old', $fileName);
            $newPath = $directoryName . DIRECTORY_SEPARATOR . $newFileName;
            ftp_rename($connection, $filePath, $newPath);

            ftp_close($connection);

            return $file;
        } catch (Exception $e) {
            return $e;
        }
    }

    /**
     * @param $file
     * @return Exception|SimpleXMLElement
     */
    private function loadXMLFromFile($file)
    {
        try {
            return simplexml_load_string($file);
        } catch (Exception $e) {
            return $e;
        }
    }

    /**
     * @param SimpleXMLElement $xmlElement
     * @return mixed
     */
    private function loadRatesFromXMLElement(SimpleXMLElement $xmlElement)
    {
        $xmlVars = get_object_vars($xmlElement);
        if (!isset($xmlVars['rate'])) {
            return false;
        }

        return $xmlVars['rate'];
    }

    /**
     * @param $rates
     * @return array
     */
    private function validateRates(array $rates)
    {
        $valid = array();
        $fieldsNumberErrorIds = array();
        $fieldsKeysErrorIds = array();
        $dateErrorIds = array();
        $numbersErrorIds = array();

        foreach ($rates as $id => $rate) {
            // fields count
            if (count($rate) < 4) {
                $fieldsNumberErrorIds[] = $id;
                continue;
            }

            // fields keys
            if (!isset($rate->loadingCity) ||
                !isset($rate->loadingCountry) ||
                !isset($rate->unloadingCity) ||
                !isset($rate->unloadingCountry) ||
                !isset($rate->cost) ||
                !isset($rate->validTo)
            ) {
                $fieldsKeysErrorIds[] = $id;
                continue;
            }

            // date
            if (!$this->validateDate($rate->validTo)) {
                $dateErrorIds[] = $id;
                continue;
            }

            // numbers
            if (!$this->validateNumber($rate->cost)) {
                $numbersErrorIds[] = $id;
                continue;
            }

            $valid[] = $rate;
        }

        return array(
            'valid' => $valid,
            'fieldsNumberErrorIds' => $fieldsNumberErrorIds,
            'fieldsKeysErrorIds' => $fieldsKeysErrorIds,
            'dateErrorIds' => $dateErrorIds,
            'numbersErrorIds' => $numbersErrorIds
        );
    }

    /**
     * @param $date
     * @return bool
     */
    private function validateDate($date)
    {
        $d = DateTime::createFromFormat('Y-m-d H:i', $date);

        return $d && $d->format('Y-m-d H:i') == $date;
    }

    /**
     * @param $number
     * @return bool
     */
    private function validateNumber($number)
    {
        $number = (float)$number;

        return $number > 0;
    }

    /**
     * @param array $rates
     * @return int
     */
    private function saveRatesToDatabase(array $rates)
    {
        $em = $this->getContainer()->get('doctrine.orm.default_entity_manager');
        $connections = $em->getRepository('AppBundle:Connection')->findNotRemoved();
        $editedCount = 0;

        foreach ($rates as $rate) {
            $connection = false;
            foreach ($connections as $key => $c) {
                if ($c->getLoading()->getName() == $rate->loadingCity &&
                    $c->getLoading()->getCountry() == $rate->loadingCountry &&
                    $c->getUnloading()->getName() == $rate->unloadingCity &&
                    $c->getUnloading()->getCountry() == $rate->unloadingCountry
                ) {
                    $connection = $c;
                    unset($connections[$key]);
                    break;
                }
            }

            // if there is connection in database
            if ($connection) {
                $oldConnection = clone $connection;
                $connection
                    ->setCost($rate->cost)
                    ->setValidTo(new \DateTime($rate->validTo));

                $connectionHistory = ConnectionController::generateConnectionHistoryObject($oldConnection, $connection);

                if ($connectionHistory) {
                    $connectionHistory
                        ->setConnection($connection);

                    $em->persist($connectionHistory);
                    $em->flush();

                    $editedCount++;
                }
            } else { // if there is no connection in database create it
                $loadingPort = $this->findOrCreatePort($rate->loadingCity, $rate->loadingCountry);
                $unloadingPort = $this->findOrCreatePort($rate->unloadingCity, $rate->unloadingCountry);

                $connection = new Connection();
                $connection
                    ->setLoading($loadingPort)
                    ->setUnloading($unloadingPort)
                    ->setCost($rate->cost)
                    ->setValidTo(new \DateTime($rate->validTo));

                $connectionHistory = new ConnectionHistory();
                $connectionHistory
                    ->setLoading($connection->getLoading())
                    ->setUnloading($connection->getUnloading())
                    ->setCost($connection->getCost())
                    ->setValidTo($connection->getValidTo())
                    ->setConnection($connection);

                $em->persist($connection);
                $em->persist($connectionHistory);
                $em->flush();
            }
        }

        // set all other connections in database not active
        $newValidTo = new \DateTime();
        $newValidTo->modify('-1 minutes');
        foreach ($connections as $connection) {
            $oldConnection = clone($connection);

            $connection->setValidTo($newValidTo);

            $connectionHistory = ConnectionController::generateConnectionHistoryObject($oldConnection, $connection);

            if ($connectionHistory) {
                $connectionHistory
                    ->setConnection($connection);

                $em->persist($connectionHistory);
                $em->flush();
            }
        }

        return $editedCount;
    }

    /**
     * @param $city
     * @param $country
     * @return Port
     */
    private function findOrCreatePort($city, $country)
    {
        $em = $this->getContainer()->get('doctrine.orm.default_entity_manager');

        if (!$this->ports) {
            $this->ports = $em->getRepository('AppBundle:Port')->findNotRemoved();
        }

        $port = false;
        foreach ($this->ports as $p) {
            if ($p->getName() == (string)$city &&
                $p->getCountry() == (string)$country
            ) {
                $port = $p;
                break;
            }
        }

        if (!$port) {
            $port = new Port();
            $port
                ->setName($city)
                ->setCountry($country);

            $portHistory = new PortHistory();
            $portHistory
                ->setName($port->getName())
                ->setCountry($port->getCountry())
                ->setPort($port);

            $em->persist($port);
            $em->persist($portHistory);
            $em->flush();
            $this->ports[] = $port;
        }

        return $port;
    }

    /**
     * @param $result
     */
    private function sendErrorMail($result)
    {
        $mailer = $this->getContainer()->get('app.service.mailer');
        $mailer->sendXMLRatesFileDownloadError($result);
    }
}
