<?php

namespace AppBundle\Validator\Constraints;

use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Security\Core\Encoder\EncoderFactory;
use Symfony\Component\Security\Core\SecurityContext;
use Doctrine\ORM\EntityManager;

class UserPasswordValidator extends ConstraintValidator
{
    /**
     * @var \Doctrine\ORM\EntityManager
     */
    private $entityManager;

    /**
     * @var EncoderFactory
     */
    private $encoderFactory;

    /**
     * @var SecurityContext
     */
    private $securityContext;

    /**
     * @param EntityManager $entityManager
     * @param EncoderFactory $encoderFactory
     * @param SecurityContext $securityContext
     */
    public function __construct(
        EntityManager $entityManager,
        EncoderFactory $encoderFactory,
        SecurityContext $securityContext
    ) {
        $this->entityManager = $entityManager;
        $this->encoderFactory = $encoderFactory;
        $this->securityContext = $securityContext;
    }

    /**
     * @param mixed $protocol
     * @param Constraint $constraint
     */
    public function validate($protocol, Constraint $constraint)
    {
        if(!$protocol->getPassword()) {
            return;
        }

        $user = $this->securityContext->getToken()->getUser();

        $newUser = clone $user;
        $this->entityManager->detach($newUser);
        $this->entityManager->refresh($user);

        $writtenPassword = $this->encoderFactory->getEncoder($user)
            ->encodePassword($newUser->getPassword(), $user->getSalt());
        $oldPassword = $user->getPassword();

        $this->entityManager->merge($newUser);

        if ($writtenPassword != $oldPassword) {
            $this->context->buildViolation($constraint->message)
                ->atPath('password')
                ->addViolation();
        }

        if(!$protocol->getNewPassword()) {
            $this->context->buildViolation($constraint->emptyNewPasswordMessage)
                ->atPath('newPassword')
                ->addViolation();
        }

        if(strlen($protocol->getNewPassword()) < 6) {
            $this->context->buildViolation($constraint->tooShortPassword)
                ->atPath('newPassword')
                ->addViolation();
        }
    }
}