<?php

namespace AppBundle\Controller\Admin;

use AppBundle\Entity\Connection;
use AppBundle\Entity\ConnectionHistory;
use AppBundle\Form\Type\ConnectionType;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;

class ConnectionController extends Controller
{
    /**
     * @Template()
     *
     * @param Request $request
     * @return array
     */
    public function listAction(Request $request)
    {
        $em = $this->getDoctrine();
        $query = $em->getRepository('AppBundle:Connection')->getListQuery();

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $query,
            $request->query->get('page', 1),
            20,
            array('defaultSortFieldName' => 'c.createdAt', 'defaultSortDirection' => 'desc')
        );

        return array(
            'pagination' => $pagination
        );
    }

    /**
     * @param Request $request
     * @return array
     */
    public function createAction(Request $request)
    {
        $connection = new Connection();
        $form = $this->createForm(new ConnectionType(), $connection, array(
            'em' => $this->getDoctrine()->getManager(),
        ));
        $form->handleRequest($request);

        if ($form->isValid()) {
            $em = $this->getDoctrine()->getManager();
            $em->persist($connection);

            $connectionHistory = new ConnectionHistory();
            $connectionHistory
                ->setLoading($connection->getLoading())
                ->setUnloading($connection->getUnloading())
                ->setCost($connection->getCost())
                ->setValidTo($connection->getValidTo())
                ->setUser($this->getUser())
                ->setConnection($connection);

            $em->persist($connectionHistory);
            $em->flush();

            $this->container->get('app.service.json')->generateJson();

            return $this->redirect($this->generateUrl('app_admin.connection.list'));
        }

        return $this->render(
            'AppBundle:Admin/Connection:form.html.twig',
            array(
                'form' => $form->createView()
            )
        );
    }

    /**
     * @param Request $request
     * @param Connection $connection
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, Connection $connection)
    {





        if($connection->getIsRemoved()) {
            throw new AccessDeniedException();
        }

        $connectionClone = clone($connection);
        $form = $this->createForm(new ConnectionType(), $connection, array(
            'em' => $this->getDoctrine()->getManager(),
        ));
        $form->handleRequest($request);

        if($request->getMethod() == 'GET') {
            $referrer = $request->headers->get('referer');
            $form->get('referrer')->setData($referrer);
        }

        if ($form->isValid()) {
            $em = $this->getDoctrine()->getManager();
            $this->saveConnectionEditHistory($connectionClone, $connection);
            $em->flush();

            $referrer = $form->get('referrer')->getData();
            $redirectTo = $referrer != NULL ? $referrer : $this->generateUrl('app_admin.connection.list');

            $this->container->get('app.service.json')->generateJson();

            return $this->redirect($redirectTo);
        }

        return $this->render(
            'AppBundle:Admin/Connection:form.html.twig',
            array(
                'connection' => $connection,
                'form' => $form->createView()
            )
        );
    }

    /**
     * @param Connection $connection
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function removeAction(Connection $connection)
    {
        if($connection->getIsRemoved()) {
            throw new AccessDeniedException();
        }

        $em = $this->getDoctrine()->getManager();
        $connection->setIsRemoved(true);

        $connectionHistory = new ConnectionHistory();
        $connectionHistory
            ->setUser($this->getUser())
            ->setConnection($connection)
            ->setIsRemoved(true);

        $em->persist($connectionHistory);
        $em->flush();

        $this->container->get('app.service.json')->generateJson();

        return $this->redirect($this->generateUrl('app_admin.connection.list'));
    }


    /**
     * @Template()
     *
     * @param Request $request
     * @param Connection $connection
     * @return array
     */
    public function historyAction(Request $request, Connection $connection)
    {
        $em = $this->getDoctrine();
        $query = $em->getRepository('AppBundle:ConnectionHistory')->getHistoryListQuery($connection);

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $query,
            $request->query->get('page', 1),
            20,
            array('defaultSortFieldName' => 'ch.createdAt', 'defaultSortDirection' => 'desc')
        );

        return array(
            'pagination' => $pagination
        );
    }

    /**
     * @Template()
     *
     * @param Request $request
     * @return array
     */
    public function removedListAction(Request $request)
    {
        $em = $this->getDoctrine();
        $query = $em->getRepository('AppBundle:Connection')->getRemovedListQuery();

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $query,
            $request->query->get('page', 1),
            20,
            array('defaultSortFieldName' => 'c.updatedAt', 'defaultSortDirection' => 'desc')
        );

        return array(
            'pagination' => $pagination
        );
    }

    public static function generateConnectionHistoryObject(Connection $oldConnection, Connection $connection)
    {
        $isEdited = false;
        $connectionHistory = new ConnectionHistory();

        if($oldConnection->getLoading() != $connection->getLoading()) {
            $connectionHistory->setLoading($connection->getLoading());
            $isEdited = true;
        }

        if($oldConnection->getUnLoading() != $connection->getUnLoading()) {
            $connectionHistory->setUnLoading($connection->getUnloading());
            $isEdited = true;
        }

        if($oldConnection->getCost() != $connection->getCost()) {
            $connectionHistory->setCost($connection->getCost());
            $isEdited = true;
        }

        if($oldConnection->getValidTo() != $connection->getValidTo()) {
            $connectionHistory->setValidTo($connection->getValidTo());
            $isEdited = true;
        }

        if(!$isEdited) {
            return false;
        }

        return $connectionHistory;
    }

    /**
     * @param Connection $oldConnection
     * @param Connection $connection
     */
    private function saveConnectionEditHistory(Connection $oldConnection, Connection $connection)
    {
        $connectionHistory = $this->generateConnectionHistoryObject($oldConnection, $connection);

        if ($connectionHistory) {
            $em = $this->getDoctrine()->getManager();
            $connectionHistory
                ->setUser($this->getUser())
                ->setConnection($connection);

            $em->persist($connectionHistory);
            $em->flush();

            $this->container->get('app.service.json')->generateJson();
        }
    }
}
