<?php

namespace AppBundle\Controller\Admin;

use AppBundle\Entity\Application;
use AppBundle\Form\Type\ApplicationType;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

class ApplicationController extends Controller
{
    /**
     * @Template()
     *
     * @param Request $request
     * @return array
     */
    public function listAction(Request $request)
    {
        $em = $this->getDoctrine();
        $query = $em->getRepository('AppBundle:Application')->getListQuery();

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $query,
            $request->query->get('page', 1),
            20,
            array('defaultSortFieldName' => 'a.createdAt', 'defaultSortDirection' => 'desc')
        );

        return array(
            'pagination' => $pagination
        );
    }

    /**
     * @Template()
     *
     * @param Request $request
     * @return array
     */
    public function createAction(Request $request)
    {
        $application = new Application();
        $form = $this->createForm(new ApplicationType(), $application, array(
            'em' => $this->getDoctrine()->getManager(),
        ));
        $form->handleRequest($request);

        if ($form->isValid()) {
            $this->countCosts($application);
            $this->setVoucher($application, $form->get('voucher')->getData());
            $em = $this->getDoctrine()->getManager();
            $em->persist($application);
            $em->flush();

            //$this->addEmailToNewsletter($application);

            $mailer = $this->get('app.service.mailer');
            $mailer->sendNewOrderUser($application);

            return $this->redirect($this->generateUrl('app_admin.application.list'));
        }

        return array(
            'form' => $form->createView(),
            'settings' => $this->getSettings(),
            'setting_euro' => $this->getSettingsEuro(),
        );
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function confirmAction(Request $request)
    {
        $em = $this->getDoctrine()->getManager();
        $application = $em->getRepository('AppBundle:Application')
            ->find($request->request->get('id'));

        if(!$application || $application->getCompletedBy()) {
            return new JsonResponse(false);
        }

        $application
            ->setCompletedBy($this->getUser())
            ->setCompletedAt(new \DateTime());
        $em->flush();

        return new JsonResponse(true);
    }

    public function commentAction(Request $request)
    {
        $em = $this->getDoctrine()->getManager();
        $application = $em->getRepository('AppBundle:Application')
            ->find($request->request->get('id'));

        if(!$application) {
            return new JsonResponse(false);
        }

        $application->setComment($request->request->get('comment'));
        $em->flush();

        return new JsonResponse(true);
    }

    /**
     * @Template()
     *
     * @param Application $application
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function detailsAction(Application $application)
    {
        return array(
            'application'
            => $application
        );
    }

    private function getSettings()
    {
        $em = $this->getDoctrine()->getManager();

        return $em->getRepository('AppBundle:Setting')->findAllData();
    }
    private function getSettingsEuro()
    {
        $em = $this->getDoctrine()->getManager();

        return $em->getRepository('AppBundle:SettingEuro')->findAllData();
    }
    private function countValue($volume, $weight)
    {
        $volume = (float)str_replace(',', '.', $volume);
        $weight = (float)str_replace(',', '.', $weight);

        $weight /= 1000;
        $value = $volume > $weight ? $volume : $weight;

        return $value < 1 ? 1 : $value;
    }

    private function countCosts(Application $application)
    {
        $em = $this->getDoctrine()->getManager();
        $connection = $em->getRepository('AppBundle:Connection')
            ->findOneByPorts($application->getLoading(), $application->getUnloading());
        $application->setConnection($connection);

        $settings = $this->getSettings();
        $value = $this->countValue($application->getVolume(), $application->getWeight());
        $weight = (float)str_replace(',', '.', $application->getWeight()) / 1000;
        $weight = $weight < 1 ? 1 : $weight;

        if($application->getDocumentary()) {
            $application->setDocumentary($settings['documentary']);
        }

        if($application->getLoading()->getCountry() == 'Polska') {
            if($application->getWarehouse()) {
                $application->setWarehouse($settings['warehouse'] * $value);
            }
        } else {
            if($application->getThc()) {
                $application->setThc($settings['thc'] * $value);
            }
            if($application->getIsps()) {
                $application->setIsps($settings['isps'] * $weight);
            }
        }
        if($application->getConnection()->getUseeuro() == 1 && $application->getUnloading()->getName() == 'Hamburg'){
          if($application->getThc() < 90){
            $application->setThc(90);
          }
        }
        if ($application->getCustoms()) {
            $application->setCustoms($settings['customs']);
        }

        $application->setConnectionCost($connection->getCost() * $value);

        if (!$application->getTransport()) {
            $application->setTransport(false);
        }

        $application->setCost(
            $application->getDocumentary() + $application->getThc() + $application->getIsps() +
            $application->getWarehouse() + $application->getConnectionCost() + $application->getCustoms()
        );

        $exchangeRate = $this->get('app.service.exchange_rate')->getAndUpdate();
        $application->setExchangeRate($exchangeRate);

        $exchangeRate_e = $this->get('app.service.exchange_rate')->getAndUpdate('e');
        $application->setExchangeRateeuro(floatval($exchangeRate_e));
    }

    private function setVoucher(Application $application, $voucher = null)
    {
        if ($voucher) {
            $em = $this->getDoctrine()->getManager();

            $voucher = $em->getRepository('AppBundle:Voucher')->findOneBy(array(
                'code' => $voucher,
                'isActive' => true
            ));

            if ($voucher) {
                $application->setVoucher($voucher);
                $application->setConnectionCost(
                    $application->getConnectionCost() * (100 - $voucher->getValue()) / 100
                );
                $application->setCost(
                    $application->getDocumentary() + $application->getThc() + $application->getIsps() +
                    $application->getWarehouse() + $application->getConnectionCost() + $application->getCustoms()
                );
            }
        }
    }

    private function addEmailToNewsletter(Application $application)
    {
        $mailChimp = $this->get('MailChimp');
        $list = $mailChimp->getList();
        $list->setDoubleOptin(false);

        $nameParts = explode(' ', $application->getFullName());
        $firstName = array_shift($nameParts);
        $lastName = implode(' ', $nameParts);

        $list->setMerge(array(
            'FNAME' => $firstName,
            'LNAME' => $lastName,
            'COMPANY' => $application->getCompany(),
            'PHONE' => $application->getPhone(),
            'LOADC' => $application->getLoading()->getCountry(),
            'LOAD' => $application->getLoading()->getName(),
            'UNLOADC' => $application->getUnloading()->getCountry(),
            'UNLOAD' => $application->getUnloading()->getName(),
            'VOLUME' => $application->getVolume(),
            'WEIGHT' => $application->getWeight() / 1000,
            'CUSTOMS' => $application->getCustoms() ? 'Yes' : 'No',
            'TRANSPORT' => $application->getTransport() ? 'Yes' : 'No'
        ));

        $list->setUpdateExisting(true);
        $list->setEmail($application->getEmail());
        $list->mergeVars();
        $list->Subscribe();
    }
}
